%=========================================================================
%
%  "Time Preference and the Distributions of Wealth and Income"
%   By Richard M. H. Suen 
%   Economic Inquiry (forthcoming)
%
%   This program is used to generate the benchmark results reported in 
%   Table 3 Panel B of the paper. 
%
%=========================================================================

close all
clear all

pause on

%=======================================
%
%  Assign parameter values.
%
%=======================================

alpha = 0.33;

theta = [0.005         0.010        0.01202        0.025        0.050       0.100      0.500];
delta = [0.07980066615 0.079845090  0.0798647146   0.080176987  0.08262995  0.0908504  0.1481711]; 

sigma = 1;
gamma = 1.022; 

N = 1000;
beta_1 = 0.966;
beta_2 = 0.992;

beta = linspace(beta_1, beta_2, N);

delta_hat = gamma - 1 + delta; 


%==============================================================================
%
%  Distribution of capital and income and their Gini coefficients.
%
%==============================================================================

r_eq = 0.11;
Kd = (alpha/r_eq)^(1/(1 - alpha));
K_Y_Ratio = Kd^(1 - alpha);

w = (1 - alpha)*(Kd^alpha);

M = length(theta);

Results = zeros(M, 11);
Errors  = zeros(M, 1);

for i = 1:M 

    LHS = (( gamma^sigma./beta - (1 + r_eq -delta(i)) )/theta(i) ).^(1/sigma);
    k   = w./( LHS - (r_eq - delta_hat(i)) );
    
    if ( sum(k > 0) < N)
        
        fprintf('Error: Capital cannot be negative!')
        theta(i)
        pause
        
    end
    
    inc = w + r_eq*k;
    
    Errors(i) = Kd - sum(k)/N;    % Errors in the capital market clearing condition.
    
    
    %=================================================================================
    % 
    %  Summary Statistics for the Distributions
    %
    %=================================================================================
    
    s1 = k./sum(k);
    s2 = inc./sum(inc);
    
    share_k   = [0 cumsum(s1)];
    share_in  = [0 cumsum(s2)];
    
    Area_k  = 0;
    Area_in = 0;
    
    for j = 1:N
        
        Area_k  = Area_k  + ( share_k(j+1) + share_k(j) )/N/2;
        Area_in = Area_in + ( share_in(j+1) + share_in(j) )/N/2;
        
    end
    
    Gini_k  = 1 - Area_k/0.5;
    Gini_in = 1 - Area_in/0.5;
    
    mean_k = sum(k)/N;
    var_k  = sum( ( k - mean_k ).^2 )/N;
    std_k  = sqrt(var_k);
    CV_k   = std_k/mean_k;
    skew_k = sum(( ( k - mean_k )/std_k ).^3)/(N-1);
    
    mean_in = sum(inc)/N;
    var_in  = sum( ( inc - mean_in ).^2 )/N;
    std_in  = sqrt(var_in);
    CV_in   = std_in/mean_in;
    skew_in = sum(( ( inc - mean_in )/std_in ).^3)/(N-1);
    
    S90 = 1 - share_k(901);
    S95 = 1 - share_k(951);
    S99 = 1 - share_k(991);
    S40 = share_k(400);
    
    T90 = 1 - share_in(901);
    T95 = 1 - share_in(951);
    T99 = 1 - share_in(991);
    T40 = share_in(400);
    
    Source = w./(w + r_eq*k);
    L1 = mean(Source(1:10));
    L2 = mean(Source(11:50));
    L3 = mean(Source(51:100));
    L4 = mean(Source(401:600));
    L5 = mean(Source(901:950));
    L6 = mean(Source(951:990));
    L7 = mean(Source(991:1000));
    
    Results(i,:) = [theta(i) Gini_in Gini_k T40 T90 T95 T99 S40 S90 S95 S99];
     
end


%==============================================
%  Results reported in Table 3 Panel B
%==============================================

Results



